<?php
/**
 * Classe de Instalação e Ativação do Plugin WooCommerce PIX Gateway
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe WC_PIX_Gateway_Installer
 */
class WC_PIX_Gateway_Installer {
    
    /**
     * Construtor
     */
    public function __construct() {
        // Hooks de ativação e desativação
        register_activation_hook(WC_PIX_GATEWAY_PLUGIN_PATH . 'woocommerce-pix-gateway.php', array($this, 'activate'));
        register_deactivation_hook(WC_PIX_GATEWAY_PLUGIN_PATH . 'woocommerce-pix-gateway.php', array($this, 'deactivate'));
        register_uninstall_hook(WC_PIX_GATEWAY_PLUGIN_PATH . 'woocommerce-pix-gateway.php', array($this, 'uninstall'));
        
        // Hook para verificar requisitos
        add_action('admin_init', array($this, 'check_requirements'));
    }
    
    /**
     * Ativar o plugin
     */
    public function activate() {
        // Verificar requisitos antes da ativação
        if (!$this->check_requirements(false)) {
            deactivate_plugins(plugin_basename(WC_PIX_GATEWAY_PLUGIN_PATH . 'woocommerce-pix-gateway.php'));
            wp_die(
                __('Este plugin requer WordPress 5.0+, WooCommerce 5.0+ e PHP 7.4+.', 'woocommerce-pix-gateway'),
                __('Requisitos não atendidos', 'woocommerce-pix-gateway'),
                array('back_link' => true)
            );
        }
        
        // Criar tabelas e opções padrão
        $this->create_default_options();
        $this->create_database_tables();
        $this->create_default_pages();
        
        // Limpar cache
        $this->clear_cache();
        
        // Registrar ativação
        $this->log_activation();
        
        // Redirecionar para página de configuração
        add_action('admin_init', array($this, 'redirect_to_settings'));
    }
    
    /**
     * Desativar o plugin
     */
    public function deactivate() {
        // Limpar cron jobs
        $this->clear_cron_jobs();
        
        // Limpar cache
        $this->clear_cache();
        
        // Registrar desativação
        $this->log_deactivation();
    }
    
    /**
     * Desinstalar o plugin
     */
    public function uninstall() {
        // Remover opções
        $this->remove_options();
        
        // Remover tabelas (opcional)
        if (get_option('wc_pix_gateway_remove_data_on_uninstall', 'no') === 'yes') {
            $this->remove_database_tables();
        }
        
        // Remover arquivos de log
        $this->remove_log_files();
    }
    
    /**
     * Verificar requisitos do sistema
     */
    public function check_requirements($show_notice = true) {
        $errors = array();
        
        // Verificar versão do WordPress
        if (version_compare(get_bloginfo('version'), WC_PIX_GATEWAY_WP_MIN_VERSION, '<')) {
            $errors[] = sprintf(
                __('WordPress %s+ é necessário. Versão atual: %s', 'woocommerce-pix-gateway'),
                WC_PIX_GATEWAY_WP_MIN_VERSION,
                get_bloginfo('version')
            );
        }
        
        // Verificar se WooCommerce está ativo
        if (!class_exists('WooCommerce')) {
            $errors[] = __('WooCommerce deve estar ativo para usar este plugin.', 'woocommerce-pix-gateway');
        } else {
            // Verificar versão do WooCommerce
            $wc_version = WC()->version;
            if (version_compare($wc_version, WC_PIX_GATEWAY_WC_MIN_VERSION, '<')) {
                $errors[] = sprintf(
                    __('WooCommerce %s+ é necessário. Versão atual: %s', 'woocommerce-pix-gateway'),
                    WC_PIX_GATEWAY_WC_MIN_VERSION,
                    $wc_version
                );
            }
        }
        
        // Verificar versão do PHP
        if (version_compare(PHP_VERSION, WC_PIX_GATEWAY_PHP_MIN_VERSION, '<')) {
            $errors[] = sprintf(
                __('PHP %s+ é necessário. Versão atual: %s', 'woocommerce-pix-gateway'),
                WC_PIX_GATEWAY_PHP_MIN_VERSION,
                PHP_VERSION
            );
        }
        
        // Verificar extensões PHP necessárias
        $required_extensions = array('curl', 'json', 'mbstring');
        foreach ($required_extensions as $ext) {
            if (!extension_loaded($ext)) {
                $errors[] = sprintf(
                    __('Extensão PHP %s é necessária.', 'woocommerce-pix-gateway'),
                    $ext
                );
            }
        }
        
        // Verificar permissões de diretório
        $upload_dir = wp_upload_dir();
        if (!is_writable($upload_dir['basedir'])) {
            $errors[] = __('O diretório de uploads não tem permissão de escrita.', 'woocommerce-pix-gateway');
        }
        
        // Mostrar notificações se necessário
        if ($show_notice && !empty($errors)) {
            $this->show_admin_notice($errors, 'error');
        }
        
        return empty($errors);
    }
    
    /**
     * Criar opções padrão
     */
    private function create_default_options() {
        $default_options = array(
            'wc_pix_gateway_enabled' => 'no',
            'wc_pix_gateway_title' => __('Pagamento PIX', 'woocommerce-pix-gateway'),
            'wc_pix_gateway_description' => __('Pague com PIX de forma rápida e segura.', 'woocommerce-pix-gateway'),
            'wc_pix_gateway_api_key' => '',
            'wc_pix_gateway_api_url' => WC_PIX_GATEWAY_API_BASE_URL,
            'wc_pix_gateway_sandbox' => 'yes',
            'wc_pix_gateway_debug' => 'no',
            'wc_pix_gateway_expiration_time' => WC_PIX_GATEWAY_DEFAULT_EXPIRATION_TIME,
            'wc_pix_gateway_remove_data_on_uninstall' => 'no',
            'wc_pix_gateway_version' => WC_PIX_GATEWAY_PLUGIN_VERSION,
            'wc_pix_gateway_installed_date' => current_time('mysql'),
            'wc_pix_gateway_activation_count' => 1
        );
        
        foreach ($default_options as $option => $value) {
            if (get_option($option) === false) {
                add_option($option, $value);
            }
        }
    }
    
    /**
     * Criar tabelas do banco de dados
     */
    private function create_database_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Tabela de logs de transações
        $table_name = $wpdb->prefix . 'wc_pix_gateway_logs';
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            transaction_id varchar(255) NOT NULL,
            order_id bigint(20) NOT NULL,
            action varchar(100) NOT NULL,
            status varchar(50) NOT NULL,
            message text,
            data longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY transaction_id (transaction_id),
            KEY order_id (order_id),
            KEY action (action),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Tabela de cache de status
        $cache_table_name = $wpdb->prefix . 'wc_pix_gateway_cache';
        
        $cache_sql = "CREATE TABLE $cache_table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            cache_key varchar(255) NOT NULL,
            cache_value longtext NOT NULL,
            expires_at datetime NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY cache_key (cache_key),
            KEY expires_at (expires_at)
        ) $charset_collate;";
        
        dbDelta($cache_sql);
    }
    
    /**
     * Criar páginas padrão
     */
    private function create_default_pages() {
        // Página de instruções PIX (opcional)
        $instructions_page_id = get_option('wc_pix_gateway_instructions_page_id');
        
        if (!$instructions_page_id) {
            $page_data = array(
                'post_title' => __('Como Pagar com PIX', 'woocommerce-pix-gateway'),
                'post_content' => $this->get_default_instructions_content(),
                'post_status' => 'publish',
                'post_type' => 'page',
                'post_name' => 'como-pagar-com-pix',
                'page_template' => 'default'
            );
            
            $page_id = wp_insert_post($page_data);
            
            if ($page_id && !is_wp_error($page_id)) {
                update_option('wc_pix_gateway_instructions_page_id', $page_id);
            }
        }
    }
    
    /**
     * Obter conteúdo padrão da página de instruções
     */
    private function get_default_instructions_content() {
        $content = '<h2>' . __('Como Pagar com PIX', 'woocommerce-pix-gateway') . '</h2>';
        $content .= '<p>' . __('O PIX é uma forma rápida e segura de pagar suas compras online.', 'woocommerce-pix-gateway') . '</p>';
        
        $content .= '<h3>' . __('Passo a Passo:', 'woocommerce-pix-gateway') . '</h3>';
        $content .= '<ol>';
        $content .= '<li>' . __('Escolha PIX como forma de pagamento no checkout', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Escaneie o QR Code com seu aplicativo bancário', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Confirme o pagamento no app', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Aguarde a confirmação automática', 'woocommerce-pix-gateway') . '</li>';
        $content .= '</ol>';
        
        $content .= '<h3>' . __('Vantagens do PIX:', 'woocommerce-pix-gateway') . '</h3>';
        $content .= '<ul>';
        $content .= '<li>' . __('Pagamento instantâneo', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Sem taxas adicionais', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Seguro e confiável', 'woocommerce-pix-gateway') . '</li>';
        $content .= '<li>' . __('Disponível 24/7', 'woocommerce-pix-gateway') . '</li>';
        $content .= '</ul>';
        
        return $content;
    }
    
    /**
     * Limpar cache
     */
    private function clear_cache() {
        // Limpar cache do WooCommerce
        if (function_exists('wc_delete_product_transients')) {
            wc_delete_product_transients();
        }
        
        // Limpar cache do WordPress
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
        
        // Limpar cache de transientes
        $this->clear_transients();
    }
    
    /**
     * Limpar transientes
     */
    private function clear_transients() {
        global $wpdb;
        
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_wc_pix_gateway_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_wc_pix_gateway_%'");
    }
    
    /**
     * Limpar cron jobs
     */
    private function clear_cron_jobs() {
        wp_clear_scheduled_hook('wc_pix_gateway_cleanup_expired_payments');
        wp_clear_scheduled_hook('wc_pix_gateway_sync_payment_status');
    }
    
    /**
     * Remover opções
     */
    private function remove_options() {
        $options = array(
            'wc_pix_gateway_enabled',
            'wc_pix_gateway_title',
            'wc_pix_gateway_description',
            'wc_pix_gateway_api_key',
            'wc_pix_gateway_api_url',
            'wc_pix_gateway_sandbox',
            'wc_pix_gateway_debug',
            'wc_pix_gateway_expiration_time',
            'wc_pix_gateway_remove_data_on_uninstall',
            'wc_pix_gateway_version',
            'wc_pix_gateway_installed_date',
            'wc_pix_gateway_activation_count',
            'wc_pix_gateway_instructions_page_id'
        );
        
        foreach ($options as $option) {
            delete_option($option);
        }
    }
    
    /**
     * Remover tabelas do banco de dados
     */
    private function remove_database_tables() {
        global $wpdb;
        
        $tables = array(
            $wpdb->prefix . 'wc_pix_gateway_logs',
            $wpdb->prefix . 'wc_pix_gateway_cache'
        );
        
        foreach ($tables as $table) {
            $wpdb->query("DROP TABLE IF EXISTS $table");
        }
    }
    
    /**
     * Remover arquivos de log
     */
    private function remove_log_files() {
        $upload_dir = wp_upload_dir();
        $log_dir = $upload_dir['basedir'] . '/wc-logs/';
        
        if (is_dir($log_dir)) {
            $files = glob($log_dir . 'pix_gateway-*.log');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }
    
    /**
     * Registrar ativação
     */
    private function log_activation() {
        $activation_count = get_option('wc_pix_gateway_activation_count', 0);
        update_option('wc_pix_gateway_activation_count', $activation_count + 1);
        update_option('wc_pix_gateway_last_activated', current_time('mysql'));
        
        // Log da ativação
        $this->log('Plugin ativado', 'info', array(
            'version' => WC_PIX_GATEWAY_PLUGIN_VERSION,
            'activation_count' => $activation_count + 1,
            'timestamp' => current_time('mysql')
        ));
    }
    
    /**
     * Registrar desativação
     */
    private function log_deactivation() {
        update_option('wc_pix_gateway_last_deactivated', current_time('mysql'));
        
        // Log da desativação
        $this->log('Plugin desativado', 'info', array(
            'timestamp' => current_time('mysql')
        ));
    }
    
    /**
     * Log de eventos
     */
    private function log($message, $level = 'info', $context = array()) {
        if (wc_pix_gateway_is_debug_enabled()) {
            $logger = wc_get_logger();
            $logger->log($level, $message, array(
                'source' => WC_PIX_GATEWAY_LOG_SOURCE,
                'context' => $context
            ));
        }
    }
    
    /**
     * Mostrar notificação administrativa
     */
    private function show_admin_notice($messages, $type = 'error') {
        if (!is_array($messages)) {
            $messages = array($messages);
        }
        
        $notice_class = 'notice notice-' . $type;
        $message_html = '<div class="' . $notice_class . '"><p>';
        $message_html .= '<strong>' . __('WooCommerce PIX Gateway:', 'woocommerce-pix-gateway') . '</strong> ';
        $message_html .= implode('<br>', $messages);
        $message_html .= '</p></div>';
        
        add_action('admin_notices', function() use ($message_html) {
            echo $message_html;
        });
    }
    
    /**
     * Redirecionar para configurações após ativação
     */
    public function redirect_to_settings() {
        if (get_option('wc_pix_gateway_do_activation_redirect', false)) {
            delete_option('wc_pix_gateway_do_activation_redirect');
            
            if (!isset($_GET['activate-multi'])) {
                wp_redirect(admin_url('admin.php?page=wc-settings&tab=checkout&section=pix_gateway'));
                exit;
            }
        }
    }
    
    /**
     * Verificar se o plugin foi atualizado
     */
    public function check_for_updates() {
        $current_version = get_option('wc_pix_gateway_version', '0.0.0');
        
        if (version_compare($current_version, WC_PIX_GATEWAY_PLUGIN_VERSION, '<')) {
            $this->update_plugin($current_version);
        }
    }
    
    /**
     * Atualizar plugin
     */
    private function update_plugin($from_version) {
        // Executar atualizações específicas de versão
        if (version_compare($from_version, '1.0.0', '<')) {
            $this->update_to_1_0_0();
        }
        
        // Atualizar versão
        update_option('wc_pix_gateway_version', WC_PIX_GATEWAY_PLUGIN_VERSION);
        update_option('wc_pix_gateway_last_updated', current_time('mysql'));
        
        // Log da atualização
        $this->log('Plugin atualizado', 'info', array(
            'from_version' => $from_version,
            'to_version' => WC_PIX_GATEWAY_PLUGIN_VERSION,
            'timestamp' => current_time('mysql')
        ));
    }
    
    /**
     * Atualizar para versão 1.0.0
     */
    private function update_to_1_0_0() {
        // Migrar configurações antigas se necessário
        $old_options = array(
            'pix_gateway_enabled' => 'wc_pix_gateway_enabled',
            'pix_gateway_title' => 'wc_pix_gateway_title',
            'pix_gateway_description' => 'wc_pix_gateway_description'
        );
        
        foreach ($old_options as $old_key => $new_key) {
            $old_value = get_option($old_key);
            if ($old_value !== false) {
                update_option($new_key, $old_value);
                delete_option($old_key);
            }
        }
    }
}

// Inicializar o instalador
new WC_PIX_Gateway_Installer();
