/**
 * JavaScript para o Gateway PIX - Frontend
 */

(function($) {
    'use strict';

    // Objeto principal do plugin
    var WCPixGateway = {
        
        init: function() {
            this.bindEvents();
            this.initCountdown();
        },
        
        bindEvents: function() {
            // Copiar código PIX
            $(document).on('click', '.copy-pix-code', this.copyPixCode);
            
            // Verificar status do pagamento
            $(document).on('click', '.check-pix-status', this.checkPixStatus);
            
            // Auto-verificação de status
            this.startAutoCheck();
        },
        
        /**
         * Copiar código PIX para área de transferência
         */
        copyPixCode: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var $textarea = $button.siblings('.pix-code-textarea');
            var originalText = $button.text();
            
            // Selecionar texto
            $textarea.select();
            $textarea.setSelectionRange(0, 99999);
            
            try {
                // Tentar copiar usando API moderna
                if (navigator.clipboard && window.isSecureContext) {
                    navigator.clipboard.writeText($textarea.val()).then(function() {
                        WCPixGateway.showMessage('Código PIX copiado com sucesso!', 'success');
                    }).catch(function() {
                        WCPixGateway.showMessage('Erro ao copiar código PIX', 'error');
                    });
                } else {
                    // Fallback para navegadores antigos
                    document.execCommand('copy');
                    WCPixGateway.showMessage('Código PIX copiado com sucesso!', 'success');
                }
            } catch (err) {
                console.error('Erro ao copiar: ', err);
                WCPixGateway.showMessage('Erro ao copiar código PIX', 'error');
            }
            
            // Feedback visual
            $button.text('Copiado!').addClass('copied');
            setTimeout(function() {
                $button.text(originalText).removeClass('copied');
            }, 2000);
        },
        
        /**
         * Verificar status do pagamento PIX
         */
        checkPixStatus: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var $container = $button.closest('.pix-payment-container');
            var orderId = $container.data('order-id');
            
            if (!orderId) {
                WCPixGateway.showMessage('ID do pedido não encontrado', 'error');
                return;
            }
            
            // Estado de loading
            $button.prop('disabled', true).text('Verificando...');
            $container.addClass('loading');
            
            // Fazer requisição AJAX
            $.ajax({
                url: wc_pix_gateway.ajax_url,
                type: 'POST',
                data: {
                    action: 'wc_pix_gateway_check_status',
                    order_id: orderId,
                    nonce: wc_pix_gateway.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var status = response.data.status;
                        
                        switch (status) {
                            case 'PAID_OUT':
                                WCPixGateway.showMessage('Pagamento confirmado! Redirecionando...', 'success');
                                WCPixGateway.updatePaymentStatus('paid');
                                
                                // Redirecionar após 2 segundos
                                setTimeout(function() {
                                    window.location.href = response.data.redirect_url || window.location.href;
                                }, 2000);
                                break;
                                
                            case 'WAITING_FOR_APPROVAL':
                                WCPixGateway.showMessage('Pagamento ainda não foi confirmado. Tente novamente em alguns instantes.', 'info');
                                break;
                                
                            case 'OVERDUE':
                                WCPixGateway.showMessage('Pagamento vencido. Entre em contato para gerar um novo PIX.', 'warning');
                                WCPixGateway.updatePaymentStatus('overdue');
                                break;
                                
                            case 'CANCELLED':
                                WCPixGateway.showMessage('Pagamento cancelado.', 'warning');
                                WCPixGateway.updatePaymentStatus('cancelled');
                                break;
                                
                            default:
                                WCPixGateway.showMessage('Status do pagamento: ' + status, 'info');
                        }
                    } else {
                        WCPixGateway.showMessage(response.data || 'Erro ao verificar status do pagamento', 'error');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Erro na requisição:', error);
                    WCPixGateway.showMessage('Erro ao verificar status. Tente novamente.', 'error');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Verificar Status');
                    $container.removeClass('loading');
                }
            });
        },
        
        /**
         * Iniciar verificação automática de status
         */
        startAutoCheck: function() {
            // Verificar a cada 30 segundos se a página estiver visível
            setInterval(function() {
                if (document.visibilityState === 'visible') {
                    var $checkButton = $('.check-pix-status');
                    if ($checkButton.length && !$checkButton.prop('disabled')) {
                        $checkButton.click();
                    }
                }
            }, 30000);
        },
        
        /**
         * Inicializar contador regressivo
         */
        initCountdown: function() {
            var $expirationElement = $('.pix-expiration .value');
            if ($expirationElement.length) {
                var expirationText = $expirationElement.text();
                var expirationDate = new Date(expirationText);
                
                if (!isNaN(expirationDate.getTime())) {
                    this.startCountdown(expirationDate);
                }
            }
        },
        
        /**
         * Iniciar contador regressivo
         */
        startCountdown: function(expirationDate) {
            var countdownInterval = setInterval(function() {
                var now = new Date().getTime();
                var distance = expirationDate.getTime() - now;
                
                if (distance < 0) {
                    clearInterval(countdownInterval);
                    $('.pix-expiration .value').text('Expirado').addClass('expired');
                    $('.check-pix-status').prop('disabled', true).text('PIX Expirado');
                    return;
                }
                
                var hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                var minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
                var seconds = Math.floor((distance % (1000 * 60)) / 1000);
                
                var timeString = '';
                if (hours > 0) timeString += hours + 'h ';
                if (minutes > 0) timeString += minutes + 'm ';
                timeString += seconds + 's';
                
                $('.pix-expiration .value').text(timeString);
                
                // Adicionar classe de aviso quando estiver próximo de expirar
                if (distance < 300000) { // 5 minutos
                    $('.pix-expiration').addClass('warning');
                }
            }, 1000);
        },
        
        /**
         * Atualizar status visual do pagamento
         */
        updatePaymentStatus: function(status) {
            var $container = $('.pix-payment-container');
            var statusClass = 'status-' + status;
            
            // Remover classes de status anteriores
            $container.removeClass('status-waiting status-paid status-overdue status-cancelled');
            
            // Adicionar nova classe de status
            $container.addClass(statusClass);
            
            // Atualizar mensagem baseada no status
            var message = '';
            var messageClass = '';
            
            switch (status) {
                case 'paid':
                    message = 'Pagamento confirmado! Seu pedido será processado em breve.';
                    messageClass = 'success';
                    break;
                case 'overdue':
                    message = 'O PIX expirou. Entre em contato para gerar um novo pagamento.';
                    messageClass = 'warning';
                    break;
                case 'cancelled':
                    message = 'Pagamento cancelado. Entre em contato para mais informações.';
                    messageClass = 'warning';
                    break;
            }
            
            if (message) {
                WCPixGateway.showMessage(message, messageClass);
            }
        },
        
        /**
         * Mostrar mensagem para o usuário
         */
        showMessage: function(message, type) {
            // Remover mensagens anteriores
            $('.pix-message').remove();
            
            // Criar nova mensagem
            var $message = $('<div class="pix-message pix-message-' + type + '">' + message + '</div>');
            
            // Adicionar ao container
            $('.pix-payment-container').prepend($message);
            
            // Mostrar com animação
            $message.hide().fadeIn(300);
            
            // Remover automaticamente após 5 segundos
            setTimeout(function() {
                $message.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        },
        
        /**
         * Gerar QR Code dinâmico (se necessário)
         */
        generateQRCode: function(text, elementId) {
            if (typeof QRCode !== 'undefined') {
                new QRCode(document.getElementById(elementId), {
                    text: text,
                    width: 250,
                    height: 250,
                    colorDark: "#000000",
                    colorLight: "#ffffff",
                    correctLevel: QRCode.CorrectLevel.H
                });
            }
        }
    };
    
    // Inicializar quando DOM estiver pronto
    $(document).ready(function() {
        WCPixGateway.init();
    });
    
    // Expor para uso global se necessário
    window.WCPixGateway = WCPixGateway;
    
})(jQuery);
