<?php
/**
 * Página de administração do Gateway PIX
 */

if (!defined('ABSPATH')) {
    exit;
}

// Verificar permissões
if (!current_user_can('manage_woocommerce')) {
    wp_die(__('Você não tem permissão para acessar esta página.', 'woocommerce-pix-gateway'));
}

// Processar formulário
if (isset($_POST['submit']) && wp_verify_nonce($_POST['wc_pix_gateway_nonce'], 'wc_pix_gateway_settings')) {
    $this->process_admin_options();
    
    echo '<div class="notice notice-success"><p>' . __('Configurações salvas com sucesso!', 'woocommerce-pix-gateway') . '</p></div>';
}

// Obter configurações atuais
$api_key = get_option('wc_pix_gateway_api_key', '');
$api_url = get_option('wc_pix_gateway_api_url', 'https://api.berapay.com.br/v1/gateway/');
$sandbox = get_option('wc_pix_gateway_sandbox', 'yes');
$debug = get_option('wc_pix_gateway_debug', 'no');
$expiration_time = get_option('wc_pix_gateway_expiration_time', 30);

// Testar conexão com a API
$test_result = '';
if (isset($_POST['test_connection']) && wp_verify_nonce($_POST['wc_pix_gateway_nonce'], 'wc_pix_gateway_test')) {
    $test_result = $this->test_api_connection();
}
?>

<div class="wrap">
    <h1><?php _e('Configurações do Gateway PIX', 'woocommerce-pix-gateway'); ?></h1>
    
    <div class="wc-pix-gateway-admin">
        <div class="wc-pix-gateway-header">
            <div class="wc-pix-gateway-logo">
                <h2><?php _e('Gateway PIX para WooCommerce', 'woocommerce-pix-gateway'); ?></h2>
                <p><?php _e('Integração com gateway de pagamento PIX usando Asaas', 'woocommerce-pix-gateway'); ?></p>
            </div>
            
            <div class="wc-pix-gateway-actions">
                <a href="<?php echo admin_url('admin.php?page=wc-settings&tab=checkout&section=pix_gateway'); ?>" 
                   class="button button-primary">
                    <?php _e('Configurações WooCommerce', 'woocommerce-pix-gateway'); ?>
                </a>
                
                <a href="https://web.berapay.com.br" target="_blank" class="button button-secondary">
                    <?php _e('Documentação da API', 'woocommerce-pix-gateway'); ?>
                </a>
            </div>
        </div>
        
        <div class="wc-pix-gateway-content">
            <div class="wc-pix-gateway-main">
                <form method="post" action="">
                    <?php wp_nonce_field('wc_pix_gateway_settings', 'wc_pix_gateway_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="api_key"><?php _e('API Key', 'woocommerce-pix-gateway'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="api_key" name="wc_pix_gateway_api_key" 
                                       value="<?php echo esc_attr($api_key); ?>" class="regular-text" />
                                <p class="description">
                                    <?php _e('Sua chave de API única fornecida pelo gateway.', 'woocommerce-pix-gateway'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="api_url"><?php _e('URL da API', 'woocommerce-pix-gateway'); ?></label>
                            </th>
                            <td>
                                <input type="url" id="api_url" name="wc_pix_gateway_api_url" 
                                       value="<?php echo esc_url($api_url); ?>" class="regular-text" />
                                <p class="description">
                                    <?php _e('URL base da API do gateway.', 'woocommerce-pix-gateway'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="expiration_time"><?php _e('Tempo de Expiração (minutos)', 'woocommerce-pix-gateway'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="expiration_time" name="wc_pix_gateway_expiration_time" 
                                       value="<?php echo esc_attr($expiration_time); ?>" min="5" max="1440" class="small-text" />
                                <p class="description">
                                    <?php _e('Tempo em minutos para o PIX expirar (5-1440 minutos).', 'woocommerce-pix-gateway'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="sandbox"><?php _e('Modo Sandbox', 'woocommerce-pix-gateway'); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="sandbox" name="wc_pix_gateway_sandbox" 
                                           value="yes" <?php checked($sandbox, 'yes'); ?> />
                                    <?php _e('Habilitar modo de teste', 'woocommerce-pix-gateway'); ?>
                                </label>
                                <p class="description">
                                    <?php _e('Use para testar pagamentos sem processar valores reais.', 'woocommerce-pix-gateway'); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="debug"><?php _e('Modo Debug', 'woocommerce-pix-gateway'); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="debug" name="wc_pix_gateway_debug" 
                                           value="yes" <?php checked($debug, 'yes'); ?> />
                                    <?php _e('Habilitar logs de debug', 'woocommerce-pix-gateway'); ?>
                                </label>
                                <p class="description">
                                    <?php _e('Registra informações detalhadas para debugging. Verifique os logs em WooCommerce > Status > Logs.', 'woocommerce-pix-gateway'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" name="submit" id="submit" 
                               class="button button-primary" 
                               value="<?php _e('Salvar Configurações', 'woocommerce-pix-gateway'); ?>" />
                    </p>
                </form>
                
                <!-- Teste de Conexão -->
                <div class="wc-pix-gateway-test-connection">
                    <h3><?php _e('Testar Conexão com a API', 'woocommerce-pix-gateway'); ?></h3>
                    <p><?php _e('Teste se sua configuração está funcionando corretamente.', 'woocommerce-pix-gateway'); ?></p>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('wc_pix_gateway_test', 'wc_pix_gateway_nonce'); ?>
                        <input type="hidden" name="test_connection" value="1" />
                        
                        <p class="submit">
                            <input type="submit" name="test_api" id="test_api" 
                                   class="button button-secondary" 
                                   value="<?php _e('Testar Conexão', 'woocommerce-pix-gateway'); ?>" />
                        </p>
                    </form>
                    
                    <?php if ($test_result): ?>
                        <div class="test-result">
                            <?php echo $test_result; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="wc-pix-gateway-sidebar">
                <div class="wc-pix-gateway-info-box">
                    <h3><?php _e('Informações do Gateway', 'woocommerce-pix-gateway'); ?></h3>
                    <ul>
                        <li><strong><?php _e('Versão:', 'woocommerce-pix-gateway'); ?></strong> <?php echo WC_PIX_GATEWAY_VERSION; ?></li>
                        <li><strong><?php _e('Status:', 'woocommerce-pix-gateway'); ?></strong> 
                            <?php if ($api_key): ?>
                                <span class="status-active"><?php _e('Ativo', 'woocommerce-pix-gateway'); ?></span>
                            <?php else: ?>
                                <span class="status-inactive"><?php _e('Inativo', 'woocommerce-pix-gateway'); ?></span>
                            <?php endif; ?>
                        </li>
                        <li><strong><?php _e('Modo:', 'woocommerce-pix-gateway'); ?></strong> 
                            <?php echo $sandbox === 'yes' ? __('Teste', 'woocommerce-pix-gateway') : __('Produção', 'woocommerce-pix-gateway'); ?>
                        </li>
                    </ul>
                </div>
                
                <div class="wc-pix-gateway-help-box">
                    <h3><?php _e('Precisa de Ajuda?', 'woocommerce-pix-gateway'); ?></h3>
                    <p><?php _e('Consulte a documentação da API ou entre em contato com o suporte.', 'woocommerce-pix-gateway'); ?></p>
                    
                    <div class="help-links">
                        <a href="https://web.berapay.com.br" target="_blank" class="button button-secondary">
                            <?php _e('Documentação', 'woocommerce-pix-gateway'); ?>
                        </a>
                        
                        <a href="mailto:suporte@seusite.com" class="button button-secondary">
                            <?php _e('Suporte', 'woocommerce-pix-gateway'); ?>
                        </a>
                    </div>
                </div>
                
                <div class="wc-pix-gateway-stats-box">
                    <h3><?php _e('Estatísticas', 'woocommerce-pix-gateway'); ?></h3>
                    <?php
                    // Contar pedidos com PIX
                    $pix_orders = wc_get_orders(array(
                        'payment_method' => 'pix_gateway',
                        'limit' => -1,
                        'return' => 'ids'
                    ));
                    
                    $total_orders = count($pix_orders);
                    $pending_orders = 0;
                    $completed_orders = 0;
                    
                    if ($total_orders > 0) {
                        foreach ($pix_orders as $order_id) {
                            $order = wc_get_order($order_id);
                            if ($order) {
                                if ($order->get_status() === 'pending') {
                                    $pending_orders++;
                                } elseif ($order->get_status() === 'completed') {
                                    $completed_orders++;
                                }
                            }
                        }
                    }
                    ?>
                    
                    <ul>
                        <li><strong><?php _e('Total de Pedidos PIX:', 'woocommerce-pix-gateway'); ?></strong> <?php echo $total_orders; ?></li>
                        <li><strong><?php _e('Aguardando Pagamento:', 'woocommerce-pix-gateway'); ?></strong> <?php echo $pending_orders; ?></li>
                        <li><strong><?php _e('Pagamentos Confirmados:', 'woocommerce-pix-gateway'); ?></strong> <?php echo $completed_orders; ?></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.wc-pix-gateway-admin {
    margin-top: 20px;
}

.wc-pix-gateway-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.wc-pix-gateway-logo h2 {
    margin: 0 0 5px 0;
    color: #2c3e50;
}

.wc-pix-gateway-logo p {
    margin: 0;
    color: #7f8c8d;
}

.wc-pix-gateway-actions {
    display: flex;
    gap: 10px;
}

.wc-pix-gateway-content {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 20px;
}

.wc-pix-gateway-main {
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.wc-pix-gateway-sidebar {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.wc-pix-gateway-info-box,
.wc-pix-gateway-help-box,
.wc-pix-gateway-stats-box {
    background: #fff;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.wc-pix-gateway-info-box h3,
.wc-pix-gateway-help-box h3,
.wc-pix-gateway-stats-box h3 {
    margin: 0 0 15px 0;
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
}

.wc-pix-gateway-info-box ul,
.wc-pix-gateway-stats-box ul {
    margin: 0;
    padding: 0;
    list-style: none;
}

.wc-pix-gateway-info-box li,
.wc-pix-gateway-stats-box li {
    margin-bottom: 10px;
    padding: 8px 0;
    border-bottom: 1px solid #f0f0f0;
}

.wc-pix-gateway-info-box li:last-child,
.wc-pix-gateway-stats-box li:last-child {
    border-bottom: none;
}

.status-active {
    color: #27ae60;
    font-weight: 600;
}

.status-inactive {
    color: #e74c3c;
    font-weight: 600;
}

.help-links {
    display: flex;
    flex-direction: column;
    gap: 10px;
    margin-top: 15px;
}

.help-links .button {
    text-align: center;
}

.wc-pix-gateway-test-connection {
    margin-top: 30px;
    padding-top: 20px;
    border-top: 2px solid #f0f0f0;
}

.test-result {
    margin-top: 15px;
    padding: 15px;
    border-radius: 6px;
    background: #f8f9fa;
    border-left: 4px solid #3498db;
}

.test-result.success {
    background: #d4edda;
    border-left-color: #28a745;
    color: #155724;
}

.test-result.error {
    background: #f8d7da;
    border-left-color: #dc3545;
    color: #721c24;
}

@media (max-width: 768px) {
    .wc-pix-gateway-header {
        flex-direction: column;
        gap: 15px;
        text-align: center;
    }
    
    .wc-pix-gateway-content {
        grid-template-columns: 1fr;
    }
    
    .wc-pix-gateway-actions {
        justify-content: center;
    }
}
</style>
