<?php
require_once 'config.php';
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *"); 
header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); 
header("Access-Control-Allow-Headers: Content-Type, Authorization"); 

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200); 
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'GET' || $_SERVER['REQUEST_METHOD'] === 'POST') {
    $requestData = [];
    
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $requestData = $_GET;
    } else {
        $requestData = json_decode(file_get_contents('php://input'), true);
    }

    // Validação dos dados obrigatórios
    if (!isset($requestData['api-key'])) {
        http_response_code(400); 
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_API_KEY')
        ]);
        exit;
    }

    // Pode consultar por ID da transação ou referência externa
    $transactionId = $requestData['idTransaction'] ?? null;
    $externalReference = $requestData['externalReference'] ?? null;
    
    if (!$transactionId && !$externalReference) {
        http_response_code(400); 
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('MISSING_REQUIRED_FIELDS')
        ]);
        exit;
    }

    include 'conectarbanco.php';
    $conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);

    if ($conn->connect_error) {
        http_response_code(500);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('DATABASE_ERROR')
        ]);
        exit;
    }

    // Validar API key
    $sql = "SELECT user_id FROM users_key WHERE api_key = ? AND status = 'ativo'";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $requestData['api-key']);
    $stmt->execute();
    $stmt->store_result();

    if ($stmt->num_rows === 0) {
        http_response_code(401);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_API_KEY')
        ]);
        $stmt->close();
        $conn->close();
        exit;
    }

    $stmt->bind_result($usuario);
    $stmt->fetch();
    $stmt->close();

    // Buscar transação
    $sql_transacao = "";
    $params = [];
    $types = "";

    if ($transactionId) {
        $sql_transacao = "SELECT * FROM solicitacoes WHERE idtransaction = ? AND user_id = ?";
        $params = [$transactionId, $usuario];
        $types = "ss";
    } else {
        $sql_transacao = "SELECT * FROM solicitacoes WHERE externalreference = ? AND user_id = ?";
        $params = [$externalReference, $usuario];
        $types = "ss";
    }

    $stmt_transacao = $conn->prepare($sql_transacao);
    $stmt_transacao->bind_param($types, ...$params);
    $stmt_transacao->execute();
    $result = $stmt_transacao->get_result();

    if ($result->num_rows === 0) {
        http_response_code(404);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('TRANSACTION_NOT_FOUND')
        ]);
        $stmt_transacao->close();
        $conn->close();
        exit;
    }

    $transacao = $result->fetch_assoc();
    $stmt_transacao->close();

    // Se a transação for do Asaas, consultar status atualizado
    if ($transacao['adquirente_ref'] === 'asaas' && !empty($transacao['idtransaction'])) {
        try {
            // Buscar configurações do Asaas
            $sql_asaas = "SELECT secret_key, url FROM ad_asaas WHERE id = 1";
            $stmt_asaas = $conn->prepare($sql_asaas);
            $stmt_asaas->execute();
            $stmt_asaas->bind_result($secret_key, $url);
            $stmt_asaas->fetch();
            $stmt_asaas->close();

            if (!empty($secret_key)) {
                // Consultar status no Asaas
                $asaasResponse = asaasRequest($url, 'payments/' . $transacao['idtransaction'], 'GET', null, $secret_key);
                
                if (!empty($asaasResponse['status'])) {
                    $statusAsaas = $asaasResponse['status'];
                    $statusLocal = $transacao['status'];
                    
                    // Mapear status do Asaas para o sistema local
                    $statusMapeado = mapearStatusAsaas($statusAsaas);
                    
                    // Atualizar status no banco se for diferente
                    if ($statusMapeado !== $statusLocal) {
                        $sql_update = "UPDATE solicitacoes SET status = ? WHERE id = ?";
                        $stmt_update = $conn->prepare($sql_update);
                        $stmt_update->bind_param("si", $statusMapeado, $transacao['id']);
                        $stmt_update->execute();
                        $stmt_update->close();
                        
                        $transacao['status'] = $statusMapeado;
                    }
                    
                    // Adicionar informações do Asaas na resposta
                    $transacao['asaas_status'] = $statusAsaas;
                    $transacao['asaas_data'] = $asaasResponse;
                }
            }
        } catch (Exception $e) {
            // Se der erro ao consultar Asaas, usar status local
            $transacao['asaas_error'] = $e->getMessage();
        }
    }

    // Preparar resposta
    $response = [
        "status" => "success",
        "message" => "Status consultado com sucesso",
        "data" => [
            "idTransaction" => $transacao['idtransaction'],
            "externalReference" => $transacao['externalreference'],
            "amount" => $transacao['amount'],
            "status" => $transacao['status'],
            "client" => [
                "name" => $transacao['client_name'],
                "document" => $transacao['client_document'],
                "email" => $transacao['client_email'],
                "telefone" => $transacao['client_telefone']
            ],
            "createdAt" => $transacao['real_data'],
            "adquirente" => $transacao['adquirente_ref'],
            "taxa_cash_in" => $transacao['taxa_cash_in'],
            "deposito_liquido" => $transacao['deposito_liquido']
        ]
    ];

    // Adicionar informações específicas do Asaas se disponível
    if (isset($transacao['asaas_status'])) {
        $response['data']['asaas_status'] = $transacao['asaas_status'];
    }

    if (isset($transacao['asaas_data'])) {
        $response['data']['asaas_data'] = $transacao['asaas_data'];
    }

    echo json_encode($response);
    $conn->close();

    } else {
        http_response_code(405);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_REQUEST_METHOD')
        ]);
    }

// Função para mapear status do Asaas para o sistema local
function mapearStatusAsaas($statusAsaas) {
    $mapeamento = getConfig('TRANSACTION_STATUS_MAP', []);
    return $mapeamento[$statusAsaas] ?? $statusAsaas;
}

// Função para fazer requisições para o Asaas
function asaasRequest($baseUrl, $endpoint, $method, $data = null, $secretKey) {
    $url = $baseUrl . $endpoint;
    
    $headers = [
        'Content-Type: application/json',
        'access_token: ' . $secretKey
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, getConfig('ASAAS_TIMEOUT', 30));
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
    } elseif ($method === 'GET') {
        curl_setopt($ch, CURLOPT_HTTPGET, true);
    }

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($response === false) {
        throw new Exception('Erro na requisição cURL');
    }

    $responseData = json_decode($response, true);
    
    if ($httpCode >= 400) {
        throw new Exception('Erro na API Asaas: ' . ($responseData['message'] ?? 'Erro desconhecido'));
    }

    return $responseData;
}
?>
